<?php

/**
 * VFB Pro Session
 *
 * This is a wrapper class for WP_Session / PHP $_SESSION
 *
 * @since 2.4.3
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * VFB_Session Class
 *
 */
class VFB_Session {

	/**
	 * Holds our session data
	 *
	 * @var array
	 * @access private
	 * @since 2.4.3
	 */
	private $session = array();

	/**
	 * Whether to use PHP $_SESSION or WP_Session
	 *
	 * PHP $_SESSION is opt-in only by defining the VFB_USE_PHP_SESSIONS constant
	 *
	 * @var bool
	 * @access private
	 * @since 2.4.3
	 */
	private $use_php_sessions = false;

	/**
	 * Get things started
	 *
	 * Defines our WP_Session constants, includes the necessary libraries and
	 * retrieves the WP Session instance
	 *
	 * @since 1.5
	 */
	public function __construct() {

		if ( is_admin() && !( defined( 'DOING_AJAX' ) && DOING_AJAX ) ) {
			return; // Don't load sessions in the admin, that's silly
		}

		$this->use_php_sessions = defined( 'VFB_USE_PHP_SESSIONS' ) && VFB_USE_PHP_SESSIONS;

		if ( $this->use_php_sessions ) {
			// Use PHP SESSION (must be enabled via the VFB_USE_PHP_SESSIONS constant)
			if ( !session_id() )
				add_action( 'init', 'session_start', -2 );
		}
		else {
			// Use WP_Session (default)
			if ( !defined( 'WP_SESSION_COOKIE' ) )
				define( 'WP_SESSION_COOKIE', 'vfb_wp_session' );

			if ( !class_exists( 'Recursive_ArrayAccess' ) )
				require_once( trailingslashit( plugin_dir_path( __FILE__ ) ) . 'libraries/class-recursive-arrayaccess.php' );

			if ( !class_exists( 'WP_Session' ) ) {
				require_once( trailingslashit( plugin_dir_path( __FILE__ ) ) . 'libraries/class-wp-session.php' );
				require_once( trailingslashit( plugin_dir_path( __FILE__ ) ) . 'libraries/wp-session.php' );
			}

			add_filter( 'wp_session_expiration', array( $this, 'set_expiration_time' ), 99999 );
		}

		if ( empty( $this->session ) && ! $this->use_php_sessions ) {
			add_action( 'plugins_loaded', array( $this, 'init' ), -1 );
		}
		else {
			add_action( 'init', array( $this, 'init' ), -1 );
		}

	}

	/**
	 * Setup the WP_Session instance
	 *
	 * @access public
	 * @since 2.4.3
	 * @return void
	 */
	public function init() {

		if( $this->use_php_sessions )
			$this->session = isset( $_SESSION['vfb'] ) && is_array( $_SESSION['vfb'] ) ? $_SESSION['vfb'] : array();
		else
			$this->session = WP_Session::get_instance();

		$token = $this->get( 'vfb-token' );
		//$purchase = $this->get( 'edd_purchase' );

		if( ! empty( $token ) ) {
			$this->set_cookie();
		} else {
			$this->set_cookie( false );
		}

		return $this->session;
	}

	/**
	 * Retrieve session ID
	 *
	 * @access public
	 * @since 2.4.3
	 * @return string Session ID
	 */
	public function get_id() {
		return $this->session->session_id;
	}


	/**
	 * Retrieve a session variable
	 *
	 * @access public
	 * @since 2.4.3
	 * @param string $key Session key
	 * @return string Session variable
	 */
	public function get( $key ) {
		$key = sanitize_key( $key );
		return isset( $this->session[ $key ] ) ? maybe_unserialize( $this->session[ $key ] ) : false;
	}

	/**
	 * Set a session variable
	 *
	 * @since 2.4.3
	 *
	 * @param $key Session key
	 * @param $value Session variable
	 * @return mixed Session variable
	 */
	public function set( $key, $value ) {
		$key = sanitize_key( $key );

		if ( is_array( $value ) )
			$this->session[ $key ] = serialize( $value );
		else
			$this->session[ $key ] = $value;

		if( $this->use_php_sessions )
			$_SESSION['vfb'] = $this->session;

		return $this->session[ $key ];
	}

	/**
	 * Set a cookie to identify whether the token is empty or not
	 *
	 * This is for hosts and caching plugins to identify if caching should be disabled
	 *
	 * @access public
	 * @since 2.4.3
	 * @param string $set Whether to set or destroy
	 * @return void
	 */
	public function set_cookie( $set = true ) {
		if( ! headers_sent() ) {
			if( $set ) {
				setcookie( 'vfb_token_set', '1', time() + 30 * 60, COOKIEPATH, COOKIE_DOMAIN, false );
			}
			else {
				setcookie( 'vfb_token_set', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, false );
			}
		}
	}

	/**
	 * Force the cookie expiration time to 24 hours
	 *
	 * @access public
	 * @since 2.4.3
	 * @param int $exp Default expiration (1 hour)
	 * @return int
	 */
	public function set_expiration_time( $exp ) {
		return current_time( 'timestamp' ) + ( 30 * 60 * 24 );
	}
}